﻿Object.defineProperty(QIPortalAPI, 'deliveryCountries', {
    get: function () {
        return this._deliveryCountries;
    },
    set: function (value) {
        document.dispatchEvent(new CustomEvent('deliveryCountriesChanged', { detail: value }));
        this._deliveryCountries = value;
    }
});

Object.defineProperty(QIPortalAPI, 'userCountry', {
    get: function () {
        return this._userCountry;
    },
    set: function (value) {
        document.dispatchEvent(new CustomEvent('userCountryChanged', { detail: value }));
        this._userCountry = value;
    }
});

Object.defineProperty(QIPortalAPI, 'userRecipientCountry', {
    get: function () {
        return this._userRecipientCountry;
    },
    set: function (value) {
        document.dispatchEvent(new CustomEvent('userRecipientCountryChanged', { detail: value }));
        this._userRecipientCountry = value;
    }
});

QIPortalAPI.formatDisplayString = function (citypart, city, district, postalCode, isCity) {
    if (city === citypart) { 
        return isCity ? `${citypart}, ${postalCode}` : `${postalCode}, ${citypart}`;
    }
    return isCity ? `${citypart} (${city}), ${postalCode}` : `${postalCode}, ${citypart} (${city})`;
}

QIPortalAPI.logGeonamesLimitExceeded = async function () {
    const errorInfo = {
        Type: 'AddressAutocomplete',
        Message: 'Geonames API limit exceeded',
    };

    try {
        const response = await fetch(`${window.location.origin}/QIPortal/ErrorLogHandler`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(errorInfo),
        });

        if (!response.ok) {
            console.error(`Failed to log error: ${response.status} ${response.statusText}`);
        }
    } catch (err) {
        console.error(`Failed to log error: ${err}`);
    }
}

QIPortalAPI.logGeonamesError = async function (err) {
    const errorInfo = {
        Type: 'AddressAutocomplete',
        Message: 'An error occurred while fetching Geonames data:' + err.message
    };

    try {
        const response = await fetch(`${window.location.origin}/QIPortal/ErrorLogHandler`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(errorInfo),
        });

        if (!response.ok) {
            console.error(`Failed to log error: ${response.status} ${response.statusText}`);
        }
    } catch (err) {
        console.error(`Failed to log error: ${err}`);
    }
}

QIPortalAPI.logRuianError = async function (err) {
    const errorInfo = {
        Type: 'AddressAutocomplete',
        Message: 'An error occurred while fetching RUIAN data:' + err.message
    };

    try {
        const response = await fetch(`${window.location.origin}/QIPortal/ErrorLogHandler`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(errorInfo),
        });

        if (!response.ok) {
            console.error(`Failed to log error: ${response.status} ${response.statusText}`);
        }
    } catch (err) {
        console.error(`Failed to log error: ${err}`);
    }
}


QIPortalAPI.addressAutocomplete = function (source, configData)
{
    let inputsource = replaceAndRemove(source);

    /* visual elements */
    const streetSuggestionDiv = document.querySelector(`div[id$="${inputsource}StreetSuggestionDiv"]`);
    const streetSuggestionUl = document.querySelector(`ul[id$="${inputsource}StreetSuggestionUl"]`);
    const streetInput = document.querySelector(`input[id$="${inputsource}Street"]`);

    const citySuggestionDiv = document.querySelector(`div[id$="${inputsource}CitySuggestionDiv"]`);
    const citySuggestionUl = document.querySelector(`ul[id$="${inputsource}CitySuggestionUl"]`);
    const cityInput = document.querySelector(`input[id$="${inputsource}City"]`);

    const zipSuggestionDiv = document.querySelector(`div[id$="${inputsource}ZIPCodeSuggestionDiv"]`);
    const zipSuggestionUl = document.querySelector(`ul[id$="${inputsource}ZIPCodeSuggestionUl"]`);
    const zipInput = document.querySelector(`input[id$="${inputsource}ZIPCode"]`);

    const countryInput = document.querySelector(`input[id$="${inputsource}Country"]`);
    const countryDropDownInput = document.querySelector(`select[id$="${inputsource}Country_DropDown"]`);
    
    let selectDisabled;
    if (countryInput.hasAttribute('readonly'))
    {
        selectDisabled = true;

    }
    
    let isenabled = isEnabled();
    countryInput.style.display = "none";

    /*config data */
    const numberOfSuggestions = configData.NumberOfSuggestions;
    const StartSuggestingCharacterCount = configData.StartSuggestingCharacterCount;
    const StartSuggestingTimeElapsedMs = configData.StartSuggestingTimeElapsedMs;

    let deliveryCountries = {};
    let defaultCountry = configData.DefaultCountry;
    let defaultUserCountry;

    let countryCode = defaultCountry.Code;

    let addressData = {
        streetName: '',
        cityName: '',
        postalCode: ''
    };
    const inputToVariableMap = {
        [streetInput.id]: 'streetName',
        [cityInput.id]: 'cityName',
        [zipInput.id]: 'postalCode'
    };

    const inputFields = [streetInput, cityInput, zipInput];
    let visibleSuggestionsUl = null;
    let countryUpdated = false;

    let currentIndex = -1;
    let items = [];
    let lastClickedElement = null;

    let isRegistration = countryInput.id.includes("User");

    addClassToULs();
    if (countryInput !== null)
    {
        populateDeliveryCountry();
    }


    /* event listeners */
    document.addEventListener('deliveryCountriesChanged', function (e)
    {
        deliveryCountries = e.detail;
        RemoveDeliveryCountries();
        populateDeliveryCountry();
    });
    if (QIPortalAPI.deliveryCountries)
    {
        deliveryCountries = QIPortalAPI.deliveryCountries;
        RemoveDeliveryCountries();
        populateDeliveryCountry();
    }

    document.addEventListener('userCountryChanged', function (e)
    {

        populateDefaultUserCountry(e.detail);
    });
    if (QIPortalAPI.userCountry)
    {
        populateDefaultUserCountry(QIPortalAPI.userCountry);
    }

    document.addEventListener('userRecipientCountryChanged', function (e)
    {
        populateDefaultRecipientCountry(e.detail);
    });
    if (QIPortalAPI.userRecipientCountry)
    {
        populateDefaultRecipientCountry(QIPortalAPI.userRecipientCountry);
    }

    countryDropDownInput.addEventListener('change', function ()
    {
        const selectedIndex = this.selectedIndex;
        const selectedOption = this.options[selectedIndex];
        countryUpdated = true;
        countryInput.value = selectedOption.getAttribute("data-inputvalue");
    });

    //click event listener
    if (document.readyState === 'loading')
    {
        document.addEventListener('DOMContentLoaded', function ()
        {
            bindInputEvent(streetInput, cityInput, zipInput, countryInput);
        });
    }
    else
    {
        bindInputEvent(streetInput, cityInput, zipInput, countryInput);
    }


    //fix for focus and blur event listener on click
    document.addEventListener('mousedown', function (event)
    {
        lastClickedElement = event.target;
    });
    //focus and blur event listener funcionality
    inputFields.forEach(inputField =>
    {
        inputField.addEventListener('focus', function ()
        {
            inputField.addEventListener('keydown', handleKeyDown);
        });
        inputField.addEventListener('blur', function ()
        {
            if (lastClickedElement && lastClickedElement.tagName === 'LI')
            {
                liClick(lastClickedElement);
            }
            inputField.removeEventListener('keydown', handleKeyDown);
            hideAllSuggestions();
        });
    });

    //country dropdown event listener
    if (countryInput !== null)
    {
        countryDropDownInput.addEventListener("change", function (event)
        {
            let selectedOption = event.target.options[event.target.selectedIndex];
            countryInput.value = selectedOption.getAttribute("data-inputvalue");
            countryCode = selectedOption.value;
        });
    }

    //on focus event listener functionality
    function handleKeyDown(event)
    {
        if (event.key === 'Enter')
        {
            if (visibleSuggestionsUl !== null)
            {
                liClick(items[currentIndex]);
            }
            event.preventDefault();
        }

        if (event.key === 'Escape')
        {
            hideAllSuggestions();
            event.preventDefault();
        }

        if (event.key === 'ArrowDown')
        {
            if (currentIndex > -1)
            {
                items[currentIndex].classList.remove("onhover");
            }

            if (currentIndex < items.length - 1)
            {
                currentIndex++;
            }
            items[currentIndex].classList.add("onhover");
            event.preventDefault();
        }

        if (event.key === 'ArrowUp')
        {
            if (currentIndex > -1)
            {
                items[currentIndex].classList.remove("onhover");
            }

            if (currentIndex > 0)
            {
                currentIndex--;
            }
            items[currentIndex].classList.add("onhover");
            event.preventDefault();
        }
    }

    function liClick(clickedLi)
    {
        const dataType = clickedLi.getAttribute('SuggestionType');

        if (dataType.includes('City'))
        {
            cityLiClick(clickedLi);
        }
        else
            if (dataType.includes('ZIP'))
            {
                zipLiClick(clickedLi);
            }
            else
                if (dataType.includes('Street'))
                {
                    streetLiClick(clickedLi);
                }

        hideAllSuggestions();
        QIPortalAPI.checkFieldValidators();
    }

    function cityLiClick(clickedLi)
    {
        const city = clickedLi.getAttribute('data-city');
        let zipCode = clickedLi.getAttribute('data-postalcode');

        if (countryCode === "CZ" || countryCode === "SK")
        {
            if (/^\d{5}$/.test(zipCode))
            {
                zipCode = zipCode.slice(0, 3) + ' ' + zipCode.slice(3);
            }
        }

        cityInput.value = city;
        zipInput.value = zipCode;

        addressData['cityName'] = city;
        addressData['postalCode'] = zipCode;

    }

    function zipLiClick(clickedLi)
    {
        const cityPart = clickedLi.getAttribute('data-city');
        let zipCode = clickedLi.getAttribute('data-postalcode');

        if (countryCode === "CZ" || countryCode === "SK")
        {
            if (/^\d{5}$/.test(zipCode))
            {
                zipCode = zipCode.slice(0, 3) + ' ' + zipCode.slice(3);
            }
        }

        zipInput.value = zipCode;
        cityInput.value = cityPart;

        addressData['cityName'] = '';
        addressData['postalCode'] = zipCode;
    }

    function streetLiClick(clickedLi)
    {
        const city = clickedLi.getAttribute('data-city');
        let zipCode = clickedLi.getAttribute('data-postalcode');
        const inText = clickedLi.innerText;

        if (countryCode === "CZ" || countryCode === "SK")
        {
            if (/^\d{5}$/.test(zipCode))
            {
                zipCode = zipCode.slice(0, 3) + ' ' + zipCode.slice(3);
            }
        }

        streetInput.value = inText.split(',')[0];
        cityInput.value = city;
        addressData['cityName'] = city;
        zipInput.value = zipCode;
        addressData['postalCode'] = zipCode;


    }

    function updateAddressData()
    {
        addressData['cityName'] = cityInput.value;
        addressData['postalCode'] = zipInput.value;
        addressData['streetName'] = streetInput.value;
    }

    /*
        this function handles all posible inputs from keyboard and set correct value
     */
    function bindInputEvent(...inputs)
    {
        inputs.forEach(Input =>
        {
            if (Input === null)
            {
                return;
            }
            let timer;

            if (Input)
            {
                Input.setAttribute('data-last-value', Input.value);

                Input.addEventListener("input", function ()
                {
                    const lastValue = this.getAttribute('data-last-value');
                    const currentValue = this.value;
                    if (currentValue !== lastValue)
                    {
                        clearTimeout(timer);
                        if (checkIfStartSuggesting(currentValue.length, getType(this.id)))
                        {
                            timer = setTimeout(() =>
                            {
                                fetchInputData(this.id, currentValue);
                            }, StartSuggestingTimeElapsedMs);
                        }
                        else
                            if (currentValue.length === 0)
                            {
                                const globalVarName = this.id;
                                if (inputToVariableMap[globalVarName])
                                {
                                    addressData[inputToVariableMap[globalVarName]] = currentValue;
                                }
                            }

                        this.setAttribute('data-last-value', currentValue);
                        const globalVarName = this.id;
                        if (inputToVariableMap[globalVarName])
                        {
                            addressData[inputToVariableMap[globalVarName]] = currentValue;
                        }
                        updateAddressData();
                    }
                });

            }
        });
    }

    /* fetch data */
    async function fetchInputData(id, value)
    {
        hideAllSuggestions();
        if (!isenabled)
        {
            return;
        }
        try
        {
            let suggestions = [];
            if (id === streetInput.id)
            {
                if (countryCode.toLowerCase() === "cz")
                {
                    suggestions = await fetchRuianData();
                }
                setSuggestions(value, suggestions, streetSuggestionDiv);
            }

            else
                if (id === zipInput.id)
                {
                    const inputDetails = {postalCode: value};
                    suggestions = await fetchZipData(inputDetails);
                    setSuggestions(value, suggestions, zipSuggestionDiv);
                }

                else
                    if (id === cityInput.id)
                    {
                        const inputDetails = {cityName: value};
                        suggestions = await fetchCityData(inputDetails);
                        setSuggestions(value, suggestions, citySuggestionDiv);
                    }
        } catch (err)
        {
        }
    }

    async function fetchRuianData()
    {
        try
        {
            let input = addressData['streetName'];
            const firstPartOfCityName = addressData.cityName.split('-')[0].trim();
            if (addressData['cityName'] !== '')
            {
                input = input + '* ' + firstPartOfCityName;
            }
            else
                if (addressData['postalCode'] !== '')
                {
                    let zipwithoutspace = addressData['postalCode'].replace(/\s+/g, '');
                    input = zipwithoutspace + ' ' + input + '*';
                }

            let url = `http://ags.cuzk.cz/arcgis/rest/services/RUIAN/Vyhledavaci_sluzba_nad_daty_RUIAN/MapServer/exts/GeocodeSOE/tables/1/suggest?text=${encodeURIComponent(input)}&f=pjson`;
            let response = await fetch(url);
            if (!response.ok)
            {
                QIPortalAPI.logRuianError(response.status);
                return;
            }

            let data = await response.json();
            if (data.suggestions.length < 1)
            {
                url = `http://ags.cuzk.cz/arcgis/rest/services/RUIAN/Vyhledavaci_sluzba_nad_daty_RUIAN/MapServer/exts/GeocodeSOE/tables/1/suggest?text=${encodeURIComponent(addressData.streetName)}&f=pjson`;
                response = await fetch(url);
                data = await response.json();
            }
            let result = await parseRuianStreetData(data);
            return result;
        } catch (err)
        {
            QIPortalAPI.logRuianError(err);
            return;
        }
    }

    async function fetchCityData(input)
    {
        try
        {
            const cityname = input.cityName.replace(/-/g, '');
            const url = QIPortalAPI.appPath + `GeoNamesProxy/fetchCityData?cityName=${encodeURIComponent(cityname)}&countryCode=${encodeURIComponent(countryCode)}`;
            const response = await fetch(url);
            if (!response.ok)
            {
                QIPortalAPI.logGeonamesError(response.status);
            }
            const data = await response.json();
            return parseZipCityData(data, true);
        } catch (err)
        {

            if (error.message.includes('HTTP error! status: 429'))
            {
                QIPortalAPI.logGeonamesLimitExceeded();
            }
            else
            {
                QIPortalAPI.logGeonamesError(err);
            }
        }
    }

    async function fetchZipData(input)
    {
        try
        {
            input = prepareInputForFetch(input);
            const {postalCode} = input;
            const url = QIPortalAPI.appPath + `GeoNamesProxy/fetchZipData?postalCode=${encodeURIComponent(postalCode)}&countryCode=${encodeURIComponent(countryCode)}`;
            const response = await fetch(url);
            if (!response.ok)
            {
                QIPortalAPI.logGeonamesError('An error occurred while fetching GeoNames data:' + response.status);
            }
            const data = await response.json();
            return parseZipCityData(data, false);
        } catch (err)
        {
            if (error.message.includes('HTTP error! status: 429'))
            {
                QIPortalAPI.logGeonamesLimitExceeded();
            }
            else
            {
                QIPortalAPI.logGeonamesError('An error occurred while fetching GeoNames data:' + err);
            }
        }
    }

    function prepareInputForFetch(input)
    {
        let {postalCode} = input;
        postalCode = postalCode.replace(/\s+/g, '');
        if (postalCode.length > 3)
        {
            postalCode = postalCode.slice(0, 3) + " " + postalCode.slice(3);
        }
        return {
            ...input,
            postalCode
        };
    }

    /*parse fetched data */
    function parseZipCityData(data, isCity)
    {
        if (data.postalCodes && data.postalCodes.length > 0)
        {
            let dataWithoutDuplicities;
            if (isCity)
            {
                dataWithoutDuplicities = removeDuplicatesCity(data.postalCodes);
            }
            else
            {
                dataWithoutDuplicities = removeDuplicatesZip(data.postalCodes);
            }
            const newZipWithCities = dataWithoutDuplicities.map(entry =>
            {
                const postalCode = entry.postalCode;
                let city = entry.placeName;
                if (city.includes("-"))
                {
                    city = city.replace("-", " - ");
                }
                const displayStringWithZip = QIPortalAPI.formatDisplayString(city, entry.adminName2 || city, entry.adminName1, postalCode, isCity);

                return {postalCode, city, displayString: displayStringWithZip};
            });
            return isCity ? getNSuggestions(newZipWithCities, "city") : getNSuggestions(newZipWithCities, "zip");

        }
        else
        {
            throw new Error('No matching results found for the provided postal code.');
        }

        function removeDuplicatesCity(data)
        {
            const uniqueEntries = {};

            data.forEach(entry =>
            {
                const postalCode = entry.postalCode;
                const adminName2 = entry.adminName2;
                const placeName = entry.placeName;
                const key = `${postalCode}-${adminName2}`;

                if (!uniqueEntries[key] || uniqueEntries[key].placeName.length < placeName.length)
                {
                    uniqueEntries[key] = entry;
                }
            });
            return Object.values(uniqueEntries);
        }

        function removeDuplicatesZip(data)
        {
            const uniqueEntries = {};
            const splitableEntries = {};
            data.forEach(entry =>
            {
                const key = entry.placeName;

                if (key.includes("-"))
                {
                    const parts = key.split("-");
                    const cleanedpart = parts[1].trim();
                    if (uniqueEntries[cleanedpart])
                    {
                        delete uniqueEntries[cleanedpart];
                        uniqueEntries[key] = entry;
                        splitableEntries[cleanedpart] = entry;
                    }
                    else
                    {
                        uniqueEntries[key] = entry;
                        splitableEntries[cleanedpart] = entry;
                    }
                }
                else
                {
                    if (!splitableEntries[key])
                    {
                        uniqueEntries[key] = entry;
                    }
                }
            });
            return Object.values(uniqueEntries);
        }
    }

    function parseRuianStreetData(data)
    {
        if (data.suggestions && data.suggestions.length > 0)
        {
            const streetsWithDetails = data.suggestions.map(entry =>
            {
                const fullText = entry.text;
                const parts = fullText.split(',').map(s => s.trim());
                let postalCode, city;

                if (parts.length === 3)
                {
                    [_, postalCode, city] = parts;
                    if (isNaN(Number(postalCode)))
                    {
                        [_, cityPartWithoutCity, lastpart] = parts;
                        partsoflastpart = lastpart.split(' ');
                        postalCode = partsoflastpart[0];
                        city = partsoflastpart[1] + ' - ' + cityPartWithoutCity;
                    }
                }
                else
                {
                    [_, postalCode] = parts;
                    city = postalCode.split(' ')[1];
                    postalCode = postalCode.split(' ')[0];
                }
                return {postalCode, city, displayString: fullText};
            });

            return getNSuggestions(streetsWithDetails, "street");
        }
    }

    /* ul handling */
    function addClassToULs()
    {
        citySuggestionUl.classList.add('suggestionList');
        streetSuggestionUl.classList.add('suggestionList');
        zipSuggestionUl.classList.add('suggestionList');
    }

    function setSuggestions(value, suggestions, element)
    {
        removeAllLiFromUl(element.firstChild);
        if (value.length > 0)
        {
            addSuggestionsToUl(element.firstChild, suggestions);
            displaySuggestions(element);
        }
    }

    function displaySuggestions(element)
    {
        hideAllSuggestions();
        if (element.children.length > 0)
        {
            element.style.visibility = 'visible';
            visibleSuggestionsUl = element;
        }
    }

    function addSuggestionsToUl(ulElement, suggestions)
    {
        items = [];
        suggestions.forEach(suggestion =>
        {
            const {displayString, postalCode, city} = suggestion;

            const listItem = createLi(displayString, ulElement.id, postalCode, city);

            ulElement.appendChild(listItem);
            items.push(listItem);
        });
    }

    function createLi(suggestion, suggestionType, postalCode, city)
    {
        const li = document.createElement('li');
        li.innerHTML = suggestion;
        li.classList.add('suggestionListItem');
        li.setAttribute('SuggestionType', suggestionType);

        li.setAttribute('data-postalcode', postalCode);
        li.setAttribute('data-city', city);
        li.addEventListener('click', function (event)
        {
            liClick(this);
        });
        return li;
    }

    function removeAllLiFromUl(ulElement)
    {
        while (ulElement.firstChild)
        {
            ulElement.removeChild(ulElement.firstChild);
        }
    }

    function hideAllSuggestions()
    {
        streetSuggestionDiv.style.visibility = 'hidden';
        citySuggestionDiv.style.visibility = 'hidden';
        zipSuggestionDiv.style.visibility = 'hidden';
        visibleSuggestionsUl = null;
    }

    function getNSuggestions(suggestions, type)
    {
        if (type === 'street' && streetSuggestionDiv.hasAttribute('data-itemscount'))
        {
            return suggestions.slice(0, parseInt(streetSuggestionDiv.dataset.itemscount));
        }
        if (type === 'city' && citySuggestionDiv.hasAttribute('data-itemscount'))
        {
            return suggestions.slice(0, parseInt(citySuggestionDiv.dataset.itemscount));
        }
        if (type === 'zip' && zipSuggestionDiv.hasAttribute('data-itemscount'))
        {
            return suggestions.slice(0, parseInt(zipSuggestionDiv.dataset.itemscount));
        }
        return suggestions.slice(0, numberOfSuggestions);
    }


    /* init handling */
    function replaceAndRemove(str)
    {
        let newStr = str.replace(/\./g, '_');

        const substringsToRemove = ['Street', 'City', 'Country', 'ZIPCode'];
        substringsToRemove.forEach((substring) =>
        {
            const regExp = new RegExp(substring, 'g');
            newStr = newStr.replace(regExp, '');
        });

        return newStr;
    }

    function populateDefaultUserCountry(userCountry)
    {
        if (!countryInput.id.includes('Recipient'))
        {
            if (getCountryType(userCountry.InputValue) !== 0)
            {
                defaultUserCountry = document.createElement("option");
                defaultUserCountry.value = userCountry.Code;
                defaultUserCountry.textContent = userCountry.DisplayName;

                let InputValue = userCountry.InputValue.slice(0, -1) + '4';
                defaultUserCountry.setAttribute('data-inputvalue', InputValue);
                selectOption(defaultUserCountry);
                countryDropDownInput.insertBefore(defaultUserCountry, countryDropDownInput.options[1]);
            }
            else
            {
                connectUserCountry(userCountry.InputValue);
            }
        }
    }

    function populateDefaultRecipientCountry(recipientCountry)
    {
        if (countryInput.id.includes('Recipient'))
        {
            if (getCountryType(recipientCountry.InputValue) !== 0)
            {
                defaultUserCountry = document.createElement("option");
                defaultUserCountry.value = recipientCountry.Code;
                defaultUserCountry.textContent = recipientCountry.DisplayName;

                let InputValue = recipientCountry.InputValue.slice(0, -1) + '4';
                defaultUserCountry.setAttribute('data-inputvalue', InputValue);
                selectOption(defaultUserCountry);
                countryDropDownInput.insertBefore(defaultUserCountry, countryDropDownInput.options[1]);
            }
            else
            {
                connectUserCountry(recipientCountry.InputValue);

            }
        }
    }

    function connectUserCountry(userInputCountry)
    {
        enableSelect();
        let modifiedUserInputCountry = userInputCountry.slice(0, -2);
        let options = countryDropDownInput.querySelectorAll('option');
        for (let option of options)
        {
            let modinput = option.getAttribute('data-inputvalue').slice(0, -2);
            if (modinput === modifiedUserInputCountry)
            {
                selectOption(option);
                break;
            }
        }
        disableSelect();
    }

    function selectOption(option)
    {
        enableSelect();
        option.selected = true;
        countryInput.value = option.getAttribute("data-inputvalue");
        if (getCountryType(option.getAttribute('data-inputvalue')) === 0)
        {
            countryInput.value = countryInput.value.slice(0, -1) + '4';
        }
        disableSelect();
    }

    function getCountryType(str)
    {
        if (str.endsWith('_0'))
        {
            return 0;
        }
        else
        if (str.endsWith('_1'))
        {
            return 1;
        }
        else
        if (str.endsWith('_2'))
        {
            return 2;
        }

        return -1;
    }

    function populateDeliveryCountry()
    {
        if (Object.keys(deliveryCountries).length === 0)
        {
            return;
        }
        deliveryCountries.sort(function (a, b)
        {
            return (a.DisplayName ?? '').localeCompare(b.DisplayName ?? '');
        });

        deliveryCountries.forEach(function (DeliveryCountry)
        {
            let option = document.createElement('option');
            option.innerText = DeliveryCountry.DisplayName;
            option.value = DeliveryCountry.Code;
            option.setAttribute('data-inputvalue', DeliveryCountry.InputValue);
            if (DeliveryCountry.IsDefault === true)
            {
                defaultCountry = DeliveryCountry;
                if (countryInput.value === '')
                {
                    option.selected = true;
                    countryInput.value = DeliveryCountry.ISOName;
                    if (countryInput.getAttribute('data-last-value') === '')
                        countryInput.setAttribute('data-last-value', countryInput.value);
                }
            }
            countryDropDownInput.appendChild(option);

            if (DeliveryCountry.InputValue === countryInput.value || checkISOName(DeliveryCountry))
            {
                option.selected = true;
            }
        });

        generateBlankCountry();
    }

    function generateBlankCountry()
    {
        let option = document.createElement('option');
        option.innerText = '';
        option.setAttribute('data-inputvalue', " ");
        if (deliveryCountries.length === 0)
        {
            option.innerText = defaultCountry.DisplayName;
        }
        option.value = defaultCountry.Code;
        countryDropDownInput.insertBefore(option, countryDropDownInput.options[0]);

        if (countryInput.value.length === 0)
        {
            selectOption(option);
        }
    }

    function RemoveDeliveryCountries()
    {
        countryDropDownInput.innerHTML = '';
    }

    function removeAutocomplete()
    {
        streetInput.setAttribute('autocomplete', 'nope');
        cityInput.setAttribute('autocomplete', 'nope');
        zipInput.setAttribute('autocomplete', 'nope');
        if (countryInput !== null)
        {
            countryInput.setAttribute('autocomplete', 'nope');
        }
    }

    function checkIfStartSuggesting(valueLen, type)
    {
        if (type === 'street' && streetSuggestionDiv.hasAttribute('data-DisplayFromCharCount'))
        {
            return valueLen >= parseInt(streetSuggestionDiv.dataset.displayfromcharcount);
            
        }
        if (type === 'city' && citySuggestionDiv.hasAttribute('data-DisplayFromCharCount'))
        {
            return valueLen >= parseInt(citySuggestionDiv.dataset.displayfromcharcount);
            
        }
        if (type === 'zip' && zipSuggestionDiv.hasAttribute('data-DisplayFromCharCount'))
        {
            return valueLen >= parseInt(zipSuggestionDiv.dataset.displayfromcharcount);
        }

        return valueLen >= StartSuggestingCharacterCount;
    }

    function getType(id)
    {
        if (id.includes('Street'))
        {
            return "street";
        }
        if (id.includes('City'))
        {
            return "city";
        }
        if (id.includes('ZIPCode'))
        {
            return "zip";
        }
        return "unknown";
    }

    function isEnabled()
    {
        if (streetSuggestionDiv.hasAttribute("data-enabled"))
        {
            if (!stringToBool(streetSuggestionDiv.dataset.enabled))
            {
                return false;
            }
        }
        if (citySuggestionDiv.hasAttribute("data-enabled"))
        {
            if (!stringToBool(citySuggestionDiv.dataset.enabled))
            {
                return false;
            }
        }
        if (zipSuggestionDiv.hasAttribute("data-enabled"))
        {
            if (!stringToBool(zipSuggestionDiv.dataset.enabled))
            {
                return false;
            }
        }
        if (!configData.AddressAutocompleteEnabled)
        {
            return false;
        }

        removeAutocomplete();
        return true;
    }

    function disableSelect()
    {
        if (selectDisabled)
        {
            countryDropDownInput.disabled = true;
        }
    }

    function enableSelect()
    {
        if (selectDisabled)
        {
            countryDropDownInput.disabled = false;
        }
    }

    function stringToBool(str)
    {
        return str === "true";
    }

    function checkISOName(DeliveryCountry)
    {
        if (countryInput.value === DeliveryCountry.ISOName)
        {
            countryInput.value = DeliveryCountry.InputValue;
            return true;
        }
        return false;
    }
};